/**
* \file: PointerListener.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <string.h>
#include <linux/input.h>
#include <adit_logging.h>
#include <pthread_adit.h>
#include "PointerListener.h"

LOG_IMPORT_CONTEXT(aauto_input)

namespace adit { namespace aauto
{

using namespace std;

struct wl_pointer_listener PointerListener::Listener;
bool PointerListener::staticInitialized = false;

PointerListener::PointerListener(InputReport& inInputReport, int inWidth, int inHeight, 
            bool inVerbose) : report(inInputReport)
{
    x = 0;
    y = 0;
    buttons = 0;

    width = inWidth;
    height = inHeight;
    verbose = inVerbose;

    if (!staticInitialized)
    {
        memset(&Listener, 0, sizeof(struct wl_pointer_listener));
        Listener.enter = handleEnter;
        Listener.leave = handleLeave;
        Listener.motion = handleMotion;
        Listener.button = handleButton;
        Listener.axis = handleAxis;

        staticInitialized = true;
    }
}

PointerListener::~PointerListener()
{
    // do nothing
}

void PointerListener::handleEnter(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
        struct wl_surface* inSurface, wl_fixed_t inX, wl_fixed_t inY)
{
    (void)inPointer;
    (void)inSerial;
    (void)inSurface;

    auto me = static_cast<PointerListener*>(inData);

    me->x = (float)wl_fixed_to_double(inX) / me->height;
    me->y = (float)wl_fixed_to_double(inY) / me->width;
    
}

void PointerListener::handleLeave(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
        struct wl_surface* inSurface)
{
    (void)inData;
    (void)inPointer;
    (void)inSerial;
    (void)inSurface;

    // do nothing

}

void PointerListener::handleMotion(void* inData, struct wl_pointer* inPointer, uint32_t inTime,
                    wl_fixed_t inX, wl_fixed_t inY)
{
    (void)inPointer;
    (void)inTime;

    auto me = static_cast<PointerListener*>(inData);

    me->x = (float)wl_fixed_to_double(inX) / me->width;
    me->y = (float)wl_fixed_to_double(inY) / me->height;

    if (me->buttons != 0)
    {
        if (me->verbose)
        {
            LOGD_VERBOSE((aauto_input, "PointerListener Motion at %.3fx%.3f", me->x, me->y));
        }

        me->report.setTouch(0, me->x, me->y, TActionMotion);
        me->report.SendInput();
    }
}

void PointerListener::handleButton(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
                    uint32_t inTime, uint32_t inButton, uint32_t inState)
{
    (void)inPointer;
    (void)inSerial;
    (void)inTime;

    auto me = static_cast<PointerListener*>(inData);

    uint8_t button = 0;
    switch (inButton)
    {
    case BTN_TOUCH:
        button |= 1;
        break;
    case BTN_LEFT:
        button |= 1 << 1;
        break;
    case BTN_RIGHT:
        button |= 1 << 2;
        break;
    case BTN_MIDDLE:
        button |= 1 << 3;
        break;
    }

    bool previouslyPressed = (me->buttons != 0);

    if (inState == 0)
    {
        // remove all up bits
        me->buttons &= ~button;
    }
    else
    {
        // add down bits
        me->buttons |= button;
    }

    if (!previouslyPressed && me->buttons != 0)
    {
        // first button was pressed
        if (me->verbose)
        {
            LOGD_VERBOSE((aauto_input, "PointerListener button down at %.3fx%.3f", me->x, me->y));
        }

        me->report.setTouch(0, me->x, me->y, TActionDown);
        me->report.SendInput();
    }
    else if (previouslyPressed && me->buttons == 0)
    {
        // last button was released
        if (me->verbose)
        {
            LOGD_VERBOSE((aauto_input, "PointerListener button up at %.3fx%.3f", me->x, me->y));
        }

        me->report.setTouch(0, 0, 0, TActionUp);
        me->report.SendInput();
    }
    else
    {
        // no change, no SendInput
    }
}

void PointerListener::handleAxis(void* inData, struct wl_pointer* inPointer, uint32_t inTime,
        uint32_t inAxis, wl_fixed_t inValue)
{
    (void)inData;
    (void)inPointer;
    (void)inTime;
    (void)inAxis;
    (void)inValue;
}

} } // namespace adit { namespace aauto
